package cc.smarnet.firmwareupdate;

import androidx.annotation.RequiresApi;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import android.Manifest;
import android.app.ProgressDialog;
import android.content.BroadcastReceiver;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.provider.DocumentsContract;
import android.provider.MediaStore;
import android.util.Log;
import android.view.View;
import android.widget.TextView;
import android.widget.Toast;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import cc.smarnet.firmwareupdate.bluetooth.BluetoothDeviceList;
import cc.smarnet.firmwareupdate.bluetooth.DeviceConnFactoryManager;
import cc.smarnet.firmwareupdate.firmware.Firmware;
import cc.smarnet.firmwareupdate.firmware.FirmwareResult;

import static cc.smarnet.firmwareupdate.bluetooth.DeviceConnFactoryManager.ACTION_PRINT_FINISH;
import static cc.smarnet.firmwareupdate.bluetooth.DeviceConnFactoryManager.ACTION_QUERY_PRINTER_STATE;
import static cc.smarnet.firmwareupdate.bluetooth.DeviceConnFactoryManager.CONN_STATE_FAILED;

public class MainActivity extends AppCompatActivity {

    private static final int SELECT_FIRMWARE = 100;
    private static final int BLUETOOTH_REQUEST_CODE = 0x001;
    private static final int REQUEST_CODE = 0x004;
    private static final String TAG = "MainActivity";
    private String[] permissions = {
            Manifest.permission.WRITE_EXTERNAL_STORAGE,
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION,
            Manifest.permission.BLUETOOTH
    };
    private DeviceConnFactoryManager deviceConnFactoryManager;
    private TextView tvConnState;
    private File file;
    private ProgressDialog progressDialog;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        tvConnState = findViewById(R.id.tv_connState);
        requestPermission(checkPermission());
    }

    @Override
    protected void onStart() {
        super.onStart();
        IntentFilter filter = new IntentFilter();
        filter.addAction(ACTION_QUERY_PRINTER_STATE);
        filter.addAction(DeviceConnFactoryManager.ACTION_CONN_STATE);
        registerReceiver(receiver, filter);
    }

    @Override
    protected void onStop() {
        super.onStop();
        unregisterReceiver(receiver);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (isConnected()) {
            closeport();
        }
    }

    private List<String> checkPermission() {
        ArrayList<String> permissionList = new ArrayList<>();
        for (String permission : permissions) {
            if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(this, permission)) {
                permissionList.add(permission);
            }
        }
        return permissionList;
    }

    private void requestPermission(List<String> per) {
        if (per.size() > 0) {
            String[] p = new String[per.size()];
            ActivityCompat.requestPermissions(this, per.toArray(p), REQUEST_CODE);
        }
    }

    public void btnConnectBluetooth(View view) {
        startActivityForResult(new Intent(this, BluetoothDeviceList.class), BLUETOOTH_REQUEST_CODE);
    }

    public void btnSelectFirmware(View view) {
        Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
        intent.setType("*/*");//设置类型，我这里是任意类型，任意后缀的可以这样写。
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        startActivityForResult(intent, SELECT_FIRMWARE);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode == RESULT_OK) {
            if (requestCode == BLUETOOTH_REQUEST_CODE) {
                closeport();
                //Get bluetoot mac address
                String macAddress = data.getStringExtra(BluetoothDeviceList.EXTRA_DEVICE_ADDRESS);
                //Init DeviceConnFactoryManager
                deviceConnFactoryManager = new DeviceConnFactoryManager.Build()
                        //Set bluetooth mac address
                        .setMacAddress(macAddress)
                        .build(this);
                //Open port
                new Thread(() -> deviceConnFactoryManager.openPort()).start();
            } else if (requestCode == SELECT_FIRMWARE) {
                Uri uri = data.getData();
                String path = getPath(this, uri);
                Log.e(TAG, "path - " + path);
                file = new File(path);
            }
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    public String getPath(final Context context, final Uri uri) {

        final boolean isKitKat = Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT;

        // DocumentProvider
        if (isKitKat && DocumentsContract.isDocumentUri(context, uri)) {
            // ExternalStorageProvider
            if (isExternalStorageDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                if ("primary".equalsIgnoreCase(type)) {
                    return Environment.getExternalStorageDirectory() + "/" + split[1];
                }
            }
            // DownloadsProvider
            else if (isDownloadsDocument(uri)) {
                final String id = DocumentsContract.getDocumentId(uri);
                final Uri contentUri = ContentUris.withAppendedId(
                        Uri.parse("content://downloads/public_downloads"), Long.valueOf(id));

                return getDataColumn(context, contentUri, null, null);
            }
            // MediaProvider
            else if (isMediaDocument(uri)) {
                final String docId = DocumentsContract.getDocumentId(uri);
                final String[] split = docId.split(":");
                final String type = split[0];

                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }

                final String selection = "_id=?";
                final String[] selectionArgs = new String[]{split[1]};

                return getDataColumn(context, contentUri, selection, selectionArgs);
            }
        }
        // MediaStore (and general)
        else if ("content".equalsIgnoreCase(uri.getScheme())) {
            return getDataColumn(context, uri, null, null);
        }
        // File
        else if ("file".equalsIgnoreCase(uri.getScheme())) {
            return uri.getPath();
        }
        return null;
    }

    public String getDataColumn(Context context, Uri uri, String selection,
                                String[] selectionArgs) {

        Cursor cursor = null;
        final String column = "_data";
        final String[] projection = {column};

        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs,
                    null);
            if (cursor != null && cursor.moveToFirst()) {
                final int column_index = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(column_index);
            }
        } finally {
            if (cursor != null)
                cursor.close();
        }
        return null;
    }


    public boolean isExternalStorageDocument(Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    public boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    public boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    private void closeport() {
        if (isConnected()) {
            deviceConnFactoryManager.closePort();
        }
    }

    private boolean isConnected() {
        return deviceConnFactoryManager != null && deviceConnFactoryManager.getConnState();
    }

    public void btnUpdateFirmware(View view) {
        if (progressDialog == null) {
            progressDialog = new ProgressDialog(this);
            progressDialog.setTitle("升级固件");
            progressDialog.setCancelable(false);
            progressDialog.setCanceledOnTouchOutside(false);
            progressDialog.setProgressStyle(ProgressDialog.STYLE_HORIZONTAL);
        }
        new Thread(() -> {
            if (file != null && file.exists()) {
                if (isConnected()) {
                    Firmware firmware = new Firmware();
                    firmware.updateFirmware(file, deviceConnFactoryManager, new FirmwareResult() {
                        @Override
                        public void verificationFail() {
                            runOnUiThread(() -> Toast.makeText(MainActivity.this, "升级包校验失败！", Toast.LENGTH_SHORT).show());
                        }

                        @Override
                        public void startUpdate() {
                            runOnUiThread(() -> {
                                progressDialog.setMax(100);
                                progressDialog.setProgress(0);
                                progressDialog.show();
                            });
                        }

                        @Override
                        public void progress(int progress) {
                            progressDialog.setProgress(progress);
                        }

                        @Override
                        public void updateSuccess() {
                            runOnUiThread(() -> {
                                progressDialog.dismiss();
                                Toast.makeText(MainActivity.this, "升级包发送成功。请长按打印机电源键进行升级。", Toast.LENGTH_SHORT).show();
                            });
                        }

                        @Override
                        public void updateFail() {
                            runOnUiThread(() -> {
                                progressDialog.dismiss();
                                Toast.makeText(MainActivity.this, "升级包发送失败。", Toast.LENGTH_SHORT).show();
                            });
                        }
                    });
                } else {
                    runOnUiThread(() -> Toast.makeText(this, "请先连接蓝牙", Toast.LENGTH_SHORT).show());
                }
            } else {
                runOnUiThread(() -> Toast.makeText(this, "请先选择升级文件", Toast.LENGTH_SHORT).show());
            }
        }).start();
    }

    private BroadcastReceiver receiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            switch (action) {
                case DeviceConnFactoryManager.ACTION_CONN_STATE:
                    int state = intent.getIntExtra(DeviceConnFactoryManager.STATE, -1);
                    int deviceId = intent.getIntExtra(DeviceConnFactoryManager.DEVICE_ID, -1);
                    switch (state) {
                        case DeviceConnFactoryManager.CONN_STATE_DISCONNECT:
                            tvConnState.setText(getString(R.string.str_conn_state_disconnect));
                            break;
                        case DeviceConnFactoryManager.CONN_STATE_CONNECTING:
                            tvConnState.setText(getString(R.string.str_conn_state_connecting));
                            break;
                        case DeviceConnFactoryManager.CONN_STATE_CONNECTED:
                            tvConnState.setText(getString(R.string.str_conn_state_connected) + "\n" + getConnDeviceInfo());
                            break;
                        case CONN_STATE_FAILED:
                            Toast.makeText(MainActivity.this, getString(R.string.str_conn_fail), Toast.LENGTH_SHORT).show();
                            tvConnState.setText(getString(R.string.str_conn_state_disconnect));
                            break;
                        default:
                            break;
                    }
                    break;
                case ACTION_PRINT_FINISH:
                    Toast.makeText(context, "Print finish！", Toast.LENGTH_SHORT).show();
                    break;
                default:
                    break;
            }
        }
    };

    private String getConnDeviceInfo() {
        String str = "";
        if (deviceConnFactoryManager != null && deviceConnFactoryManager.getConnState()) {
            str += "BLUETOOTH\n";
            str += "MacAddress: " + deviceConnFactoryManager.getMacAddress();
        }
        return str;
    }
}